/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.quickstart;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import org.eclipse.jetty.util.resource.Resource;

public class AttributeNormalizer {
    private static final Logger LOG = Log.getLogger(AttributeNormalizer.class);
    private static final Pattern __propertyPattern = Pattern.compile("(?<=[^$]|^)\\$\\{([^}]*)\\}");
    private static Comparator<Attribute> attrComparator = new Comparator<Attribute>(){

        @Override
        public int compare(Attribute o1, Attribute o2) {
            if (o1.value == null && o2.value != null) {
                return -1;
            }
            if (o1.value != null && o2.value == null) {
                return 1;
            }
            if (o1.value == null && o2.value == null) {
                return 0;
            }
            int diff = o2.value.length() - o1.value.length();
            if (diff != 0) {
                return diff;
            }
            diff = o2.value.compareTo(o1.value);
            if (diff != 0) {
                return diff;
            }
            return o2.weight - o1.weight;
        }
    };
    private URI warURI;
    private Map<String, Attribute> attributes = new HashMap<String, Attribute>();
    private List<PathAttribute> paths = new ArrayList<PathAttribute>();
    private List<URIAttribute> uris = new ArrayList<URIAttribute>();

    private static URI toCanonicalURI(URI uri) {
        String ascii = (uri = uri.normalize()).toASCIIString();
        if (ascii.endsWith("/")) {
            try {
                uri = new URI(ascii.substring(0, ascii.length() - 1));
            }
            catch (URISyntaxException e) {
                throw new IllegalArgumentException(e);
            }
        }
        return uri;
    }

    private static Path toCanonicalPath(String path) {
        if (path == null) {
            return null;
        }
        if (path.length() > 1 && path.endsWith("/")) {
            path = path.substring(0, path.length() - 1);
        }
        return AttributeNormalizer.toCanonicalPath(FileSystems.getDefault().getPath(path, new String[0]));
    }

    private static Path toCanonicalPath(Path path) {
        if (path == null) {
            return null;
        }
        if (Files.exists(path, new LinkOption[0])) {
            try {
                return path.toRealPath(new LinkOption[0]);
            }
            catch (IOException e) {
                throw new IllegalArgumentException(e);
            }
        }
        return path.toAbsolutePath();
    }

    private static void add(List<PathAttribute> paths, List<URIAttribute> uris, String key, int weight) {
        String value = System.getProperty(key);
        if (value != null) {
            Path path = AttributeNormalizer.toCanonicalPath(value);
            paths.add(new PathAttribute(key, path, weight));
            uris.add(new URIAttribute(key + ".uri", AttributeNormalizer.toCanonicalURI(path.toUri()), weight));
        }
    }

    public AttributeNormalizer(Resource baseResource) {
        if (baseResource == null) {
            throw new IllegalArgumentException("No base resource!");
        }
        this.warURI = AttributeNormalizer.toCanonicalURI(baseResource.getURI());
        if (!this.warURI.isAbsolute()) {
            throw new IllegalArgumentException("WAR URI is not absolute: " + this.warURI);
        }
        AttributeNormalizer.add(this.paths, this.uris, "jetty.base", 9);
        AttributeNormalizer.add(this.paths, this.uris, "jetty.home", 8);
        AttributeNormalizer.add(this.paths, this.uris, "user.home", 7);
        AttributeNormalizer.add(this.paths, this.uris, "user.dir", 6);
        if (this.warURI.getScheme().equalsIgnoreCase("file")) {
            this.paths.add(new PathAttribute("WAR.path", AttributeNormalizer.toCanonicalPath(new File(this.warURI).toString()), 10));
        }
        this.uris.add(new URIAttribute("WAR.uri", this.warURI, 9));
        this.uris.add(new URIAttribute("WAR", this.warURI, 8));
        Collections.sort(this.paths, attrComparator);
        Collections.sort(this.uris, attrComparator);
        Stream.concat(this.paths.stream(), this.uris.stream()).forEach(a -> this.attributes.put(a.key, (Attribute)a));
        if (LOG.isDebugEnabled()) {
            for (Attribute attr : this.attributes.values()) {
                LOG.debug(attr.toString(), new Object[0]);
            }
        }
    }

    public String normalize(Object o) {
        try {
            URI uri = null;
            Path path = null;
            if (o instanceof URI) {
                uri = AttributeNormalizer.toCanonicalURI((URI)o);
            } else if (o instanceof Resource) {
                uri = AttributeNormalizer.toCanonicalURI(((Resource)o).getURI());
            } else if (o instanceof URL) {
                uri = AttributeNormalizer.toCanonicalURI(((URL)o).toURI());
            } else if (o instanceof File) {
                path = ((File)o).getAbsoluteFile().getCanonicalFile().toPath();
            } else if (o instanceof Path) {
                path = (Path)o;
            } else {
                String s = o.toString();
                try {
                    uri = new URI(s);
                    if (uri.getScheme() == null) {
                        return s;
                    }
                }
                catch (URISyntaxException e) {
                    return s;
                }
            }
            if (uri != null) {
                if ("jar".equalsIgnoreCase(uri.getScheme())) {
                    String raw = uri.getRawSchemeSpecificPart();
                    int bang = raw.indexOf("!/");
                    String normal = this.normalize(raw.substring(0, bang));
                    String suffix = raw.substring(bang);
                    return "jar:" + normal + suffix;
                }
                if (uri.isAbsolute()) {
                    return this.normalizeUri(uri);
                }
            } else if (path != null) {
                return this.normalizePath(path);
            }
        }
        catch (Exception e) {
            LOG.warn((Throwable)e);
        }
        return String.valueOf(o);
    }

    protected String normalizeUri(URI uri) {
        for (URIAttribute a : this.uris) {
            try {
                String s;
                if (uri.compareTo(a.uri) == 0) {
                    return String.format("${%s}", a.key);
                }
                if (!a.uri.getScheme().equalsIgnoreCase(uri.getScheme()) || a.uri.getHost() == null && uri.getHost() != null || a.uri.getHost() != null && !a.uri.getHost().equals(uri.getHost())) continue;
                if (a.uri.getPath().equals(uri.getPath())) {
                    return a.value;
                }
                if (!uri.getPath().startsWith(a.uri.getPath()) || (s = uri.getPath().substring(a.uri.getPath().length())).charAt(0) != '/') continue;
                return String.format("${%s}%s", a.key, new URI(s).toASCIIString());
            }
            catch (URISyntaxException e) {
                LOG.ignore((Throwable)e);
            }
        }
        return uri.toASCIIString();
    }

    protected String normalizePath(Path path) {
        for (PathAttribute a : this.paths) {
            try {
                if (path.equals(a.path) || Files.isSameFile(path, a.path)) {
                    return String.format("${%s}", a.key);
                }
            }
            catch (IOException ignore) {
                LOG.ignore((Throwable)ignore);
            }
            if (!path.startsWith(a.path)) continue;
            return String.format("${%s}/%s", a.key, a.path.relativize(path).toString());
        }
        return path.toString();
    }

    public String expand(String str) {
        return this.expand(str, new Stack<String>());
    }

    public String expand(String str, Stack<String> seenStack) {
        if (str == null) {
            return str;
        }
        if (str.indexOf("${") < 0) {
            return str;
        }
        Matcher mat = __propertyPattern.matcher(str);
        StringBuilder expanded = new StringBuilder();
        int offset = 0;
        while (mat.find(offset)) {
            String property = mat.group(1);
            if (seenStack.contains(property)) {
                int idx;
                StringBuilder err = new StringBuilder();
                err.append("Property expansion loop detected: ");
                for (int i = idx = seenStack.lastIndexOf(property); i < seenStack.size(); ++i) {
                    err.append((String)seenStack.get(i));
                    err.append(" -> ");
                }
                err.append(property);
                throw new RuntimeException(err.toString());
            }
            seenStack.push(property);
            expanded.append(str.subSequence(offset, mat.start()));
            String value = this.getString(property);
            if (value == null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Unable to expand: {}", new Object[]{property});
                }
                expanded.append(mat.group());
            } else {
                value = this.expand(value, seenStack);
                expanded.append(value);
            }
            offset = mat.end();
        }
        expanded.append(str.substring(offset));
        if (expanded.indexOf("$$") >= 0) {
            return expanded.toString().replaceAll("\\$\\$", "\\$");
        }
        return expanded.toString();
    }

    private String getString(String property) {
        if (property == null) {
            return null;
        }
        Attribute a = this.attributes.get(property);
        if (a != null) {
            return a.value;
        }
        return System.getProperty(property);
    }

    private static class URIAttribute
    extends Attribute {
        public final URI uri;

        public URIAttribute(String key, URI uri, int weight) {
            super(key, uri.toASCIIString(), weight);
            this.uri = uri;
        }

        public String toString() {
            return String.format("URIAttribute[%s=>%s]", this.key, this.uri);
        }
    }

    private static class PathAttribute
    extends Attribute {
        public final Path path;

        public PathAttribute(String key, Path path, int weight) {
            super(key, path.toString(), weight);
            this.path = path;
        }

        public String toString() {
            return String.format("PathAttribute[%s=>%s]", this.key, this.path);
        }
    }

    private static class Attribute {
        final String key;
        final String value;
        final int weight;

        public Attribute(String key, String value, int weight) {
            this.key = key;
            this.value = value;
            this.weight = weight;
        }
    }
}

